package runtime

import (
	"bufio"
	"context"
	"fmt"
	"log"
	"math"
	"os"
	"reflect"
	"strconv"
	"strings"
	"sync"
	"time"
)

// Add performs dynamic addition for numbers, strings, and lists
func Add(a, b any) (any, error) {
	switch aVal := a.(type) {
	case int64:
		switch bVal := b.(type) {
		case int64:
			return aVal + bVal, nil
		case float64:
			return float64(aVal) + bVal, nil
		default:
			return nil, fmt.Errorf("unsupported operand types for +: 'int' and '%T'", b)
		}
	case float64:
		switch bVal := b.(type) {
		case int64:
			return aVal + float64(bVal), nil
		case float64:
			return aVal + bVal, nil
		default:
			return nil, fmt.Errorf("unsupported operand types for +: 'float' and '%T'", b)
		}
	case string:
		if bVal, ok := b.(string); ok {
			return aVal + bVal, nil
		}
		return nil, fmt.Errorf("can only concatenate str (not \"%T\") to str", b)
	case []any:
		if bVal, ok := b.([]any); ok {
			result := make([]any, len(aVal)+len(bVal))
			copy(result, aVal)
			copy(result[len(aVal):], bVal)
			return result, nil
		}
		return nil, fmt.Errorf("can only concatenate list (not \"%T\") to list", b)
	default:
		return nil, fmt.Errorf("unsupported operand types for +: '%T' and '%T'", a, b)
	}
}

// Sub performs dynamic subtraction for numbers
func Sub(a, b any) (any, error) {
	switch aVal := a.(type) {
	case int64:
		switch bVal := b.(type) {
		case int64:
			return aVal - bVal, nil
		case float64:
			return float64(aVal) - bVal, nil
		default:
			return nil, fmt.Errorf("unsupported operand types for -: 'int' and '%T'", b)
		}
	case float64:
		switch bVal := b.(type) {
		case int64:
			return aVal - float64(bVal), nil
		case float64:
			return aVal - bVal, nil
		default:
			return nil, fmt.Errorf("unsupported operand types for -: 'float' and '%T'", b)
		}
	default:
		return nil, fmt.Errorf("unsupported operand types for -: '%T' and '%T'", a, b)
	}
}

// Mul performs dynamic multiplication for numbers and string repetition
func Mul(a, b any) (any, error) {
	switch aVal := a.(type) {
	case int64:
		switch bVal := b.(type) {
		case int64:
			return aVal * bVal, nil
		case float64:
			return float64(aVal) * bVal, nil
		case string:
			if aVal < 0 {
				return "", nil
			}
			return strings.Repeat(bVal, int(aVal)), nil
		default:
			return nil, fmt.Errorf("unsupported operand types for *: 'int' and '%T'", b)
		}
	case float64:
		switch bVal := b.(type) {
		case int64:
			return aVal * float64(bVal), nil
		case float64:
			return aVal * bVal, nil
		default:
			return nil, fmt.Errorf("unsupported operand types for *: 'float' and '%T'", b)
		}
	case string:
		if bVal, ok := b.(int64); ok {
			if bVal < 0 {
				return "", nil
			}
			return strings.Repeat(aVal, int(bVal)), nil
		}
		return nil, fmt.Errorf("can't multiply sequence by non-int of type '%T'", b)
	default:
		return nil, fmt.Errorf("unsupported operand types for *: '%T' and '%T'", a, b)
	}
}

// Div performs dynamic division (always returns float)
func Div(a, b any) (any, error) {
	aFloat, err := toFloat(a)
	if err != nil {
		return nil, fmt.Errorf("unsupported operand type for /: '%T'", a)
	}
	bFloat, err := toFloat(b)
	if err != nil {
		return nil, fmt.Errorf("unsupported operand type for /: '%T'", b)
	}
	if bFloat == 0 {
		return nil, fmt.Errorf("division by zero")
	}
	return aFloat / bFloat, nil
}

// FloorDiv performs floor division
func FloorDiv(a, b any) (any, error) {
	aFloat, err := toFloat(a)
	if err != nil {
		return nil, fmt.Errorf("unsupported operand type for //: '%T'", a)
	}
	bFloat, err := toFloat(b)
	if err != nil {
		return nil, fmt.Errorf("unsupported operand type for //: '%T'", b)
	}
	if bFloat == 0 {
		return nil, fmt.Errorf("integer division or modulo by zero")
	}

	result := math.Floor(aFloat / bFloat)

	// Return int64 if both operands were integers
	if isInt(a) && isInt(b) {
		return int64(result), nil
	}
	return result, nil
}

// Mod performs modulo operation
func Mod(a, b any) (any, error) {
	switch aVal := a.(type) {
	case int64:
		switch bVal := b.(type) {
		case int64:
			if bVal == 0 {
				return nil, fmt.Errorf("integer division or modulo by zero")
			}
			return aVal % bVal, nil
		case float64:
			if bVal == 0 {
				return nil, fmt.Errorf("float modulo")
			}
			return math.Mod(float64(aVal), bVal), nil
		default:
			return nil, fmt.Errorf("unsupported operand types for %%: 'int' and '%T'", b)
		}
	case float64:
		switch bVal := b.(type) {
		case int64:
			if bVal == 0 {
				return nil, fmt.Errorf("float modulo")
			}
			return math.Mod(aVal, float64(bVal)), nil
		case float64:
			if bVal == 0 {
				return nil, fmt.Errorf("float modulo")
			}
			return math.Mod(aVal, bVal), nil
		default:
			return nil, fmt.Errorf("unsupported operand types for %%: 'float' and '%T'", b)
		}
	default:
		return nil, fmt.Errorf("unsupported operand types for %%: '%T' and '%T'", a, b)
	}
}

// Pow performs exponentiation
func Pow(a, b any) (any, error) {
	aFloat, err := toFloat(a)
	if err != nil {
		return nil, fmt.Errorf("unsupported operand type for **: '%T'", a)
	}
	bFloat, err := toFloat(b)
	if err != nil {
		return nil, fmt.Errorf("unsupported operand type for **: '%T'", b)
	}

	result := math.Pow(aFloat, bFloat)

	// Return int64 if result is a whole number and both operands were integers
	if isInt(a) && isInt(b) && result == math.Trunc(result) {
		return int64(result), nil
	}
	return result, nil
}

// Comparison operations
func Equal(a, b any) (bool, error) {
	return reflect.DeepEqual(a, b), nil
}

func NotEqual(a, b any) (bool, error) {
	eq, err := Equal(a, b)
	return !eq, err
}

func Less(a, b any) (bool, error) {
	switch aVal := a.(type) {
	case int64:
		switch bVal := b.(type) {
		case int64:
			return aVal < bVal, nil
		case float64:
			return float64(aVal) < bVal, nil
		default:
			return false, fmt.Errorf("'<' not supported between instances of 'int' and '%T'", b)
		}
	case float64:
		switch bVal := b.(type) {
		case int64:
			return aVal < float64(bVal), nil
		case float64:
			return aVal < bVal, nil
		default:
			return false, fmt.Errorf("'<' not supported between instances of 'float' and '%T'", b)
		}
	case string:
		if bVal, ok := b.(string); ok {
			return aVal < bVal, nil
		}
		return false, fmt.Errorf("'<' not supported between instances of 'str' and '%T'", b)
	default:
		return false, fmt.Errorf("'<' not supported between instances of '%T' and '%T'", a, b)
	}
}

func Greater(a, b any) (bool, error) {
	less, err := Less(a, b)
	if err != nil {
		return false, err
	}
	equal, err := Equal(a, b)
	if err != nil {
		return false, err
	}
	return !less && !equal, nil
}

func LessEqual(a, b any) (bool, error) {
	less, err := Less(a, b)
	if err != nil {
		return false, err
	}
	equal, err := Equal(a, b)
	if err != nil {
		return false, err
	}
	return less || equal, nil
}

func GreaterEqual(a, b any) (bool, error) {
	less, err := Less(a, b)
	if err != nil {
		return false, err
	}
	return !less, nil
}

// IndexList gets an element from a list
func IndexList(lst any, idx any) (any, error) {
	listVal, ok := lst.([]any)
	if !ok {
		return nil, fmt.Errorf("'%T' object is not subscriptable", lst)
	}

	idxVal, ok := idx.(int64)
	if !ok {
		return nil, fmt.Errorf("list indices must be integers, not %T", idx)
	}

	// Handle negative indices
	if idxVal < 0 {
		idxVal += int64(len(listVal))
	}

	if idxVal < 0 || idxVal >= int64(len(listVal)) {
		return nil, fmt.Errorf("list index out of range")
	}

	return listVal[idxVal], nil
}

// GetDict gets a value from a dictionary
func GetDict(m any, key string) (any, error) {
	mapVal, ok := m.(map[string]any)
	if !ok {
		return nil, fmt.Errorf("'%T' object is not subscriptable", m)
	}

	value, exists := mapVal[key]
	if !exists {
		return nil, fmt.Errorf("KeyError: '%s'", key)
	}

	return value, nil
}

// SetDict sets a value in a dictionary
func SetDict(m any, key string, v any) error {
	mapVal, ok := m.(map[string]any)
	if !ok {
		return fmt.Errorf("'%T' object does not support item assignment", m)
	}

	mapVal[key] = v
	return nil
}

// Negate performs unary negation
func Negate(a any) (any, error) {
	switch val := a.(type) {
	case int64:
		return -val, nil
	case float64:
		return -val, nil
	default:
		return nil, fmt.Errorf("bad operand type for unary -: '%T'", a)
	}
}

// Positive performs unary positive (no-op for numbers)
func Positive(a any) (any, error) {
	switch val := a.(type) {
	case int64:
		return val, nil
	case float64:
		return val, nil
	default:
		return nil, fmt.Errorf("bad operand type for unary +: '%T'", a)
	}
}

// IsTruthy determines if a value is truthy in Petroleum
func IsTruthy(obj any) bool {
	switch val := obj.(type) {
	case nil:
		return false
	case bool:
		return val
	case int64:
		return val != 0
	case float64:
		return val != 0.0
	case string:
		return val != ""
	case []any:
		return len(val) > 0
	case map[string]any:
		return len(val) > 0
	default:
		return true
	}
}

// Print is a builtin print function for Petroleum
func Print(args ...any) any {
	strs := make([]string, len(args))
	for i, arg := range args {
		strs[i] = ToString(arg)
	}
	fmt.Println(strings.Join(strs, " "))
	return nil
}

// ToString converts any value to its string representation
func ToString(obj any) string {
	switch val := obj.(type) {
	case nil:
		return "None"
	case bool:
		if val {
			return "True"
		}
		return "False"
	case int64:
		return strconv.FormatInt(val, 10)
	case float64:
		return strconv.FormatFloat(val, 'g', -1, 64)
	case string:
		return val
	case []any:
		strs := make([]string, len(val))
		for i, item := range val {
			strs[i] = ToString(item)
		}
		return "[" + strings.Join(strs, ", ") + "]"
	case map[string]any:
		pairs := make([]string, 0, len(val))
		for k, v := range val {
			pairs = append(pairs, fmt.Sprintf("'%s': %s", k, ToString(v)))
		}
		return "{" + strings.Join(pairs, ", ") + "}"
	default:
		return fmt.Sprintf("%v", val)
	}
}

// Input reads a line of input from the user
func Input(prompt ...any) string {
	if len(prompt) > 0 {
		fmt.Print(ToString(prompt[0]))
	}

	scanner := bufio.NewScanner(os.Stdin)
	if scanner.Scan() {
		return scanner.Text()
	}
	return ""
}

// Helper functions
func toFloat(val any) (float64, error) {
	switch v := val.(type) {
	case int64:
		return float64(v), nil
	case float64:
		return v, nil
	default:
		return 0, fmt.Errorf("cannot convert %T to float", val)
	}
}

func isInt(val any) bool {
	_, ok := val.(int64)
	return ok
}

// PError represents a Petroleum error with context
type PError struct {
	Kind     string
	Message  string
	Location string
	Cause    error
}

func (e *PError) Error() string {
	if e.Location != "" {
		return fmt.Sprintf("%s: %s at %s", e.Kind, e.Message, e.Location)
	}
	return fmt.Sprintf("%s: %s", e.Kind, e.Message)
}

func (e *PError) Unwrap() error {
	return e.Cause
}

// NewPError creates a new PError
func NewPError(kind, message, location string, cause error) *PError {
	return &PError{
		Kind:     kind,
		Message:  message,
		Location: location,
		Cause:    cause,
	}
}

// Ctx represents a Petroleum execution context
type Ctx struct {
	context.Context
	cancel context.CancelFunc
	wg     *sync.WaitGroup
}

// NewCtx creates a new Petroleum context
func NewCtx() *Ctx {
	ctx, cancel := context.WithCancel(context.Background())
	return &Ctx{
		Context: ctx,
		cancel:  cancel,
		wg:      &sync.WaitGroup{},
	}
}

// Cancel cancels the context
func (c *Ctx) Cancel() {
	c.cancel()
}

// Wait waits for all pets to complete
func (c *Ctx) Wait() {
	c.wg.Wait()
}

// PetTask represents a running pet (goroutine)
type PetTask struct {
	ctx    *Ctx
	result chan any
	err    chan error
	done   chan struct{}
}

// NewPetTask creates a new pet task
func NewPetTask(ctx *Ctx, fn func() (any, error)) *PetTask {
	task := &PetTask{
		ctx:    ctx,
		result: make(chan any, 1),
		err:    make(chan error, 1),
		done:   make(chan struct{}),
	}

	ctx.wg.Add(1)
	go func() {
		defer ctx.wg.Done()
		defer close(task.done)

		result, err := fn()
		if err != nil {
			task.err <- err
		} else {
			task.result <- result
		}
	}()

	return task
}

// Await waits for the pet task to complete
func (p *PetTask) Await() (any, error) {
	select {
	case result := <-p.result:
		return result, nil
	case err := <-p.err:
		return nil, err
	case <-p.ctx.Done():
		return nil, p.ctx.Err()
	}
}

// Leash represents a channel for communication between pets
type Leash struct {
	ch     chan any
	closed bool
	mu     sync.RWMutex
}

// NewLeash creates a new leash with optional buffer size
func NewLeash(bufferSize ...int) *Leash {
	size := 0
	if len(bufferSize) > 0 {
		size = bufferSize[0]
	}
	return &Leash{
		ch: make(chan any, size),
	}
}

// Send sends a value through the leash
func (l *Leash) Send(value any) error {
	l.mu.RLock()
	defer l.mu.RUnlock()

	if l.closed {
		return fmt.Errorf("cannot send on closed leash")
	}

	select {
	case l.ch <- value:
		return nil
	default:
		return fmt.Errorf("leash buffer full")
	}
}

// Receive receives a value from the leash
func (l *Leash) Receive() (any, error) {
	value, ok := <-l.ch
	if !ok {
		return nil, fmt.Errorf("leash closed")
	}
	return value, nil
}

// Close closes the leash
func (l *Leash) Close() {
	l.mu.Lock()
	defer l.mu.Unlock()

	if !l.closed {
		close(l.ch)
		l.closed = true
	}
}

// PetGroup manages a group of pets with structured concurrency
type PetGroup struct {
	ctx   *Ctx
	tasks []*PetTask
	mu    sync.Mutex
}

// NewPetGroup creates a new pet group
func NewPetGroup() *PetGroup {
	return &PetGroup{
		ctx: NewCtx(),
	}
}

// Pet starts a new pet in the group
func (pg *PetGroup) Pet(fn func() (any, error)) *PetTask {
	pg.mu.Lock()
	defer pg.mu.Unlock()

	task := NewPetTask(pg.ctx, fn)
	pg.tasks = append(pg.tasks, task)
	return task
}

// WaitAll waits for all pets in the group to complete
func (pg *PetGroup) WaitAll() error {
	pg.ctx.Wait()
	return nil
}

// Cancel cancels all pets in the group
func (pg *PetGroup) Cancel() {
	pg.ctx.Cancel()
}

// Logger provides structured logging for Petroleum
type Logger struct {
	enabled bool
	logger  *log.Logger
}

var globalLogger = &Logger{
	enabled: os.Getenv("PETROLEUM_LOG") != "",
	logger:  log.New(os.Stderr, "[PETROLEUM] ", log.LstdFlags|log.Lshortfile),
}

// LogError logs an error if logging is enabled
func LogError(err error, location string) {
	if globalLogger.enabled {
		globalLogger.logger.Printf("ERROR at %s: %v", location, err)
	}
}

// LogExec logs function execution if logging is enabled
func LogExec(funcName string, args []any, result any, err error) {
	if globalLogger.enabled {
		if err != nil {
			globalLogger.logger.Printf("EXEC %s(%v) -> ERROR: %v", funcName, args, err)
		} else {
			globalLogger.logger.Printf("EXEC %s(%v) -> %v", funcName, args, result)
		}
	}
}

// SetLogging enables or disables logging
func SetLogging(enabled bool) {
	globalLogger.enabled = enabled
}

// WithTimeout creates a context with timeout
func WithTimeout(ctx *Ctx, duration time.Duration) *Ctx {
	timeoutCtx, cancel := context.WithTimeout(ctx.Context, duration)
	return &Ctx{
		Context: timeoutCtx,
		cancel:  cancel,
		wg:      ctx.wg,
	}
}

// Contains checks if a collection contains a value (for 'in' operator)
func Contains(collection any, value any) bool {
	switch c := collection.(type) {
	case []any:
		for _, item := range c {
			if eq, err := Equal(item, value); err == nil && eq {
				return true
			}
		}
		return false
	case map[string]any:
		if key, ok := value.(string); ok {
			_, exists := c[key]
			return exists
		}
		return false
	case string:
		if s, ok := value.(string); ok {
			return strings.Contains(c, s)
		}
		return false
	default:
		return false
	}
}

// ToIterable converts a value to an iterable slice for 'for' loops
func ToIterable(v any) []any {
	switch val := v.(type) {
	case []any:
		return val
	case string:
		result := make([]any, len(val))
		for i, ch := range val {
			result[i] = string(ch)
		}
		return result
	case map[string]any:
		result := make([]any, 0, len(val))
		for k := range val {
			result = append(result, k)
		}
		return result
	default:
		return []any{}
	}
}

// Range generates a range of integers (like Python's range)
func Range(args ...any) []any {
	var start, end, step int64 = 0, 0, 1

	switch len(args) {
	case 1:
		if v, ok := args[0].(int64); ok {
			end = v
		}
	case 2:
		if v, ok := args[0].(int64); ok {
			start = v
		}
		if v, ok := args[1].(int64); ok {
			end = v
		}
	case 3:
		if v, ok := args[0].(int64); ok {
			start = v
		}
		if v, ok := args[1].(int64); ok {
			end = v
		}
		if v, ok := args[2].(int64); ok {
			step = v
		}
	}

	if step == 0 {
		return []any{}
	}

	var result []any
	if step > 0 {
		for i := start; i < end; i += step {
			result = append(result, i)
		}
	} else {
		for i := start; i > end; i += step {
			result = append(result, i)
		}
	}
	return result
}

// EnumeratedItem represents an (index, value) pair from enumerate
type EnumeratedItem struct {
	Index int64
	Value any
}

// Enumerate returns (index, value) pairs for iteration
func Enumerate(iterable any) []EnumeratedItem {
	items := ToIterable(iterable)
	result := make([]EnumeratedItem, len(items))
	for i, item := range items {
		result[i] = EnumeratedItem{Index: int64(i), Value: item}
	}
	return result
}

// Len returns the length of a collection
func Len(v any) (int64, error) {
	switch val := v.(type) {
	case string:
		return int64(len(val)), nil
	case []any:
		return int64(len(val)), nil
	case map[string]any:
		return int64(len(val)), nil
	default:
		return 0, fmt.Errorf("object of type '%T' has no len()", v)
	}
}

// ================== String Methods ==================

// StrUpper converts string to uppercase
func StrUpper(s any) (string, error) {
	if str, ok := s.(string); ok {
		return strings.ToUpper(str), nil
	}
	return "", fmt.Errorf("upper() requires string, got %T", s)
}

// StrLower converts string to lowercase
func StrLower(s any) (string, error) {
	if str, ok := s.(string); ok {
		return strings.ToLower(str), nil
	}
	return "", fmt.Errorf("lower() requires string, got %T", s)
}

// StrStrip removes leading and trailing whitespace
func StrStrip(s any) (string, error) {
	if str, ok := s.(string); ok {
		return strings.TrimSpace(str), nil
	}
	return "", fmt.Errorf("strip() requires string, got %T", s)
}

// StrLstrip removes leading whitespace
func StrLstrip(s any) (string, error) {
	if str, ok := s.(string); ok {
		return strings.TrimLeft(str, " \t\n\r"), nil
	}
	return "", fmt.Errorf("lstrip() requires string, got %T", s)
}

// StrRstrip removes trailing whitespace
func StrRstrip(s any) (string, error) {
	if str, ok := s.(string); ok {
		return strings.TrimRight(str, " \t\n\r"), nil
	}
	return "", fmt.Errorf("rstrip() requires string, got %T", s)
}

// StrSplit splits string by delimiter
func StrSplit(s any, delim any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("split() requires string, got %T", s)
	}
	d, ok := delim.(string)
	if !ok {
		return nil, fmt.Errorf("split() delimiter must be string, got %T", delim)
	}
	parts := strings.Split(str, d)
	result := make([]any, len(parts))
	for i, p := range parts {
		result[i] = p
	}
	return result, nil
}

// StrJoin joins list elements with delimiter
func StrJoin(delim any, list any) (string, error) {
	d, ok := delim.(string)
	if !ok {
		return "", fmt.Errorf("join() delimiter must be string, got %T", delim)
	}
	items, ok := list.([]any)
	if !ok {
		return "", fmt.Errorf("join() requires list, got %T", list)
	}
	strs := make([]string, len(items))
	for i, item := range items {
		strs[i] = ToString(item)
	}
	return strings.Join(strs, d), nil
}

// StrReplace replaces occurrences of old with new
func StrReplace(s, old, new any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("replace() requires string, got %T", s)
	}
	oldStr, ok := old.(string)
	if !ok {
		return "", fmt.Errorf("replace() old must be string, got %T", old)
	}
	newStr, ok := new.(string)
	if !ok {
		return "", fmt.Errorf("replace() new must be string, got %T", new)
	}
	return strings.ReplaceAll(str, oldStr, newStr), nil
}

// StrStartswith checks if string starts with prefix
func StrStartswith(s, prefix any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("startswith() requires string, got %T", s)
	}
	p, ok := prefix.(string)
	if !ok {
		return false, fmt.Errorf("startswith() prefix must be string, got %T", prefix)
	}
	return strings.HasPrefix(str, p), nil
}

// StrEndswith checks if string ends with suffix
func StrEndswith(s, suffix any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("endswith() requires string, got %T", s)
	}
	suf, ok := suffix.(string)
	if !ok {
		return false, fmt.Errorf("endswith() suffix must be string, got %T", suffix)
	}
	return strings.HasSuffix(str, suf), nil
}

// StrFind returns index of substring, -1 if not found
func StrFind(s, sub any) (int64, error) {
	str, ok := s.(string)
	if !ok {
		return -1, fmt.Errorf("find() requires string, got %T", s)
	}
	substr, ok := sub.(string)
	if !ok {
		return -1, fmt.Errorf("find() substring must be string, got %T", sub)
	}
	return int64(strings.Index(str, substr)), nil
}

// StrCount counts occurrences of substring
func StrCount(s, sub any) (int64, error) {
	str, ok := s.(string)
	if !ok {
		return 0, fmt.Errorf("count() requires string, got %T", s)
	}
	substr, ok := sub.(string)
	if !ok {
		return 0, fmt.Errorf("count() substring must be string, got %T", sub)
	}
	return int64(strings.Count(str, substr)), nil
}

// StrTitle converts to title case
func StrTitle(s any) (string, error) {
	if str, ok := s.(string); ok {
		return strings.Title(str), nil
	}
	return "", fmt.Errorf("title() requires string, got %T", s)
}

// StrCapitalize capitalizes first letter
func StrCapitalize(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("capitalize() requires string, got %T", s)
	}
	if len(str) == 0 {
		return "", nil
	}
	return strings.ToUpper(string(str[0])) + strings.ToLower(str[1:]), nil
}

// StrReverse reverses a string
func StrReverse(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("reverse() requires string, got %T", s)
	}
	runes := []rune(str)
	for i, j := 0, len(runes)-1; i < j; i, j = i+1, j-1 {
		runes[i], runes[j] = runes[j], runes[i]
	}
	return string(runes), nil
}

// StrWords splits string into words (by whitespace)
func StrWords(s any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("words() requires string, got %T", s)
	}
	parts := strings.Fields(str)
	result := make([]any, len(parts))
	for i, p := range parts {
		result[i] = p
	}
	return result, nil
}

// StrLines splits string into lines
func StrLines(s any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("lines() requires string, got %T", s)
	}
	// Handle both \n and \r\n
	normalized := strings.ReplaceAll(str, "\r\n", "\n")
	parts := strings.Split(normalized, "\n")
	result := make([]any, len(parts))
	for i, p := range parts {
		result[i] = p
	}
	return result, nil
}

// StrTruncate truncates string to max length with suffix
func StrTruncate(s any, maxLen any, suffix any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("truncate() requires string, got %T", s)
	}
	max, ok := maxLen.(int64)
	if !ok {
		return "", fmt.Errorf("truncate() maxLen must be int, got %T", maxLen)
	}
	suf := "..."
	if suffix != nil {
		if s, ok := suffix.(string); ok {
			suf = s
		}
	}
	if int64(len(str)) <= max {
		return str, nil
	}
	if max <= int64(len(suf)) {
		return suf[:max], nil
	}
	return str[:max-int64(len(suf))] + suf, nil
}

// StrCenter centers string with padding
func StrCenter(s any, width any, fill any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("center() requires string, got %T", s)
	}
	w, ok := width.(int64)
	if !ok {
		return "", fmt.Errorf("center() width must be int, got %T", width)
	}
	fillChar := " "
	if fill != nil {
		if f, ok := fill.(string); ok && len(f) > 0 {
			fillChar = string(f[0])
		}
	}
	if int64(len(str)) >= w {
		return str, nil
	}
	padding := int(w) - len(str)
	left := padding / 2
	right := padding - left
	return strings.Repeat(fillChar, left) + str + strings.Repeat(fillChar, right), nil
}

// StrPadLeft pads string on the left
func StrPadLeft(s any, width any, fill any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("pad_left() requires string, got %T", s)
	}
	w, ok := width.(int64)
	if !ok {
		return "", fmt.Errorf("pad_left() width must be int, got %T", width)
	}
	fillChar := " "
	if fill != nil {
		if f, ok := fill.(string); ok && len(f) > 0 {
			fillChar = string(f[0])
		}
	}
	if int64(len(str)) >= w {
		return str, nil
	}
	return strings.Repeat(fillChar, int(w)-len(str)) + str, nil
}

// StrPadRight pads string on the right
func StrPadRight(s any, width any, fill any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("pad_right() requires string, got %T", s)
	}
	w, ok := width.(int64)
	if !ok {
		return "", fmt.Errorf("pad_right() width must be int, got %T", width)
	}
	fillChar := " "
	if fill != nil {
		if f, ok := fill.(string); ok && len(f) > 0 {
			fillChar = string(f[0])
		}
	}
	if int64(len(str)) >= w {
		return str, nil
	}
	return str + strings.Repeat(fillChar, int(w)-len(str)), nil
}

// StrBetween extracts text between two delimiters
func StrBetween(s, start, end any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("between() requires string, got %T", s)
	}
	startStr, ok := start.(string)
	if !ok {
		return "", fmt.Errorf("between() start must be string, got %T", start)
	}
	endStr, ok := end.(string)
	if !ok {
		return "", fmt.Errorf("between() end must be string, got %T", end)
	}
	startIdx := strings.Index(str, startStr)
	if startIdx == -1 {
		return "", nil
	}
	startIdx += len(startStr)
	endIdx := strings.Index(str[startIdx:], endStr)
	if endIdx == -1 {
		return "", nil
	}
	return str[startIdx : startIdx+endIdx], nil
}

// StrMask masks characters in a string (for passwords, credit cards, etc.)
func StrMask(s any, maskChar any, keepStart any, keepEnd any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("mask() requires string, got %T", s)
	}
	mask := "*"
	if maskChar != nil {
		if m, ok := maskChar.(string); ok && len(m) > 0 {
			mask = string(m[0])
		}
	}
	start := int64(0)
	if keepStart != nil {
		if ks, ok := keepStart.(int64); ok {
			start = ks
		}
	}
	end := int64(0)
	if keepEnd != nil {
		if ke, ok := keepEnd.(int64); ok {
			end = ke
		}
	}
	if int(start+end) >= len(str) {
		return str, nil
	}
	masked := str[:start] + strings.Repeat(mask, len(str)-int(start)-int(end)) + str[len(str)-int(end):]
	return masked, nil
}

// StrSlugify converts string to URL-safe slug
func StrSlugify(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("slugify() requires string, got %T", s)
	}
	// Convert to lowercase
	result := strings.ToLower(str)
	// Replace spaces and underscores with hyphens
	result = strings.ReplaceAll(result, " ", "-")
	result = strings.ReplaceAll(result, "_", "-")
	// Remove non-alphanumeric characters (except hyphens)
	var cleaned strings.Builder
	for _, r := range result {
		if (r >= 'a' && r <= 'z') || (r >= '0' && r <= '9') || r == '-' {
			cleaned.WriteRune(r)
		}
	}
	result = cleaned.String()
	// Remove multiple consecutive hyphens
	for strings.Contains(result, "--") {
		result = strings.ReplaceAll(result, "--", "-")
	}
	// Trim leading/trailing hyphens
	result = strings.Trim(result, "-")
	return result, nil
}

// StrIsEmpty checks if string is empty or whitespace only
func StrIsEmpty(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_empty() requires string, got %T", s)
	}
	return strings.TrimSpace(str) == "", nil
}

// StrIsDigit checks if string contains only digits
func StrIsDigit(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_digit() requires string, got %T", s)
	}
	if len(str) == 0 {
		return false, nil
	}
	for _, r := range str {
		if r < '0' || r > '9' {
			return false, nil
		}
	}
	return true, nil
}

// StrIsAlpha checks if string contains only letters
func StrIsAlpha(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_alpha() requires string, got %T", s)
	}
	if len(str) == 0 {
		return false, nil
	}
	for _, r := range str {
		if !((r >= 'a' && r <= 'z') || (r >= 'A' && r <= 'Z')) {
			return false, nil
		}
	}
	return true, nil
}

// StrIsAlnum checks if string contains only alphanumeric characters
func StrIsAlnum(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_alnum() requires string, got %T", s)
	}
	if len(str) == 0 {
		return false, nil
	}
	for _, r := range str {
		if !((r >= 'a' && r <= 'z') || (r >= 'A' && r <= 'Z') || (r >= '0' && r <= '9')) {
			return false, nil
		}
	}
	return true, nil
}

// StrIndent adds prefix to each line
func StrIndent(s any, prefix any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("indent() requires string, got %T", s)
	}
	p, ok := prefix.(string)
	if !ok {
		return "", fmt.Errorf("indent() prefix must be string, got %T", prefix)
	}
	lines := strings.Split(str, "\n")
	for i, line := range lines {
		if line != "" {
			lines[i] = p + line
		}
	}
	return strings.Join(lines, "\n"), nil
}

// StrDedent removes common leading whitespace from all lines
func StrDedent(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("dedent() requires string, got %T", s)
	}
	lines := strings.Split(str, "\n")
	// Find minimum indentation (ignoring empty lines)
	minIndent := -1
	for _, line := range lines {
		if strings.TrimSpace(line) == "" {
			continue
		}
		indent := 0
		for _, r := range line {
			if r == ' ' {
				indent++
			} else if r == '\t' {
				indent += 4
			} else {
				break
			}
		}
		if minIndent == -1 || indent < minIndent {
			minIndent = indent
		}
	}
	if minIndent <= 0 {
		return str, nil
	}
	// Remove the common indentation
	for i, line := range lines {
		if strings.TrimSpace(line) == "" {
			continue
		}
		removed := 0
		j := 0
		for j < len(line) && removed < minIndent {
			if line[j] == ' ' {
				removed++
				j++
			} else if line[j] == '\t' {
				removed += 4
				j++
			} else {
				break
			}
		}
		lines[i] = line[j:]
	}
	return strings.Join(lines, "\n"), nil
}

// StrWrap wraps text to specified width
func StrWrap(s any, width any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("wrap() requires string, got %T", s)
	}
	w, ok := width.(int64)
	if !ok {
		return "", fmt.Errorf("wrap() width must be int, got %T", width)
	}
	if w <= 0 {
		return str, nil
	}
	words := strings.Fields(str)
	if len(words) == 0 {
		return "", nil
	}
	var lines []string
	currentLine := words[0]
	for _, word := range words[1:] {
		if int64(len(currentLine)+1+len(word)) <= w {
			currentLine += " " + word
		} else {
			lines = append(lines, currentLine)
			currentLine = word
		}
	}
	lines = append(lines, currentLine)
	return strings.Join(lines, "\n"), nil
}

// StrOrdinalize converts number to ordinal string (1st, 2nd, 3rd, etc.)
func StrOrdinalize(n any) (string, error) {
	var num int64
	switch v := n.(type) {
	case int64:
		num = v
	case float64:
		num = int64(v)
	default:
		return "", fmt.Errorf("ordinalize() requires number, got %T", n)
	}
	suffix := "th"
	if num%100 < 11 || num%100 > 13 {
		switch num % 10 {
		case 1:
			suffix = "st"
		case 2:
			suffix = "nd"
		case 3:
			suffix = "rd"
		}
	}
	return fmt.Sprintf("%d%s", num, suffix), nil
}

// StrPluralize returns singular or plural form based on count
func StrPluralize(word any, count any, plural any) (string, error) {
	w, ok := word.(string)
	if !ok {
		return "", fmt.Errorf("pluralize() word must be string, got %T", word)
	}
	var n int64
	switch v := count.(type) {
	case int64:
		n = v
	case float64:
		n = int64(v)
	default:
		return "", fmt.Errorf("pluralize() count must be number, got %T", count)
	}
	if n == 1 {
		return w, nil
	}
	// Use provided plural or generate one
	if plural != nil {
		if p, ok := plural.(string); ok {
			return p, nil
		}
	}
	// Simple English pluralization rules
	if strings.HasSuffix(w, "y") && len(w) > 1 {
		lastChar := w[len(w)-2]
		if lastChar != 'a' && lastChar != 'e' && lastChar != 'i' && lastChar != 'o' && lastChar != 'u' {
			return w[:len(w)-1] + "ies", nil
		}
	}
	if strings.HasSuffix(w, "s") || strings.HasSuffix(w, "x") || strings.HasSuffix(w, "z") ||
		strings.HasSuffix(w, "ch") || strings.HasSuffix(w, "sh") {
		return w + "es", nil
	}
	return w + "s", nil
}

// StrRepeat repeats a string n times
func StrRepeat(s any, n any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("repeat() requires string, got %T", s)
	}
	count, ok := n.(int64)
	if !ok {
		return "", fmt.Errorf("repeat() count must be int, got %T", n)
	}
	if count < 0 {
		return "", nil
	}
	return strings.Repeat(str, int(count)), nil
}

// StrSqueeze replaces consecutive occurrences of a character with a single one
func StrSqueeze(s any, char any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("squeeze() requires string, got %T", s)
	}
	ch := " "
	if char != nil {
		if c, ok := char.(string); ok && len(c) > 0 {
			ch = string(c[0])
		}
	}
	for strings.Contains(str, ch+ch) {
		str = strings.ReplaceAll(str, ch+ch, ch)
	}
	return str, nil
}

// StrRemove removes all occurrences of substring
func StrRemove(s any, sub any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("remove() requires string, got %T", s)
	}
	substr, ok := sub.(string)
	if !ok {
		return "", fmt.Errorf("remove() substring must be string, got %T", sub)
	}
	return strings.ReplaceAll(str, substr, ""), nil
}

// ================== Python Parity Methods ==================

// StrSwapcase swaps uppercase to lowercase and vice versa
func StrSwapcase(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("swapcase() requires string, got %T", s)
	}
	var result strings.Builder
	for _, r := range str {
		if r >= 'a' && r <= 'z' {
			result.WriteRune(r - 32)
		} else if r >= 'A' && r <= 'Z' {
			result.WriteRune(r + 32)
		} else {
			result.WriteRune(r)
		}
	}
	return result.String(), nil
}

// StrZfill pads string with zeros on the left
func StrZfill(s any, width any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("zfill() requires string, got %T", s)
	}
	w, ok := width.(int64)
	if !ok {
		return "", fmt.Errorf("zfill() width must be int, got %T", width)
	}
	if int64(len(str)) >= w {
		return str, nil
	}
	// Handle negative numbers
	if len(str) > 0 && (str[0] == '-' || str[0] == '+') {
		return string(str[0]) + strings.Repeat("0", int(w)-len(str)) + str[1:], nil
	}
	return strings.Repeat("0", int(w)-len(str)) + str, nil
}

// StrRfind returns index of last occurrence of substring, -1 if not found
func StrRfind(s, sub any) (int64, error) {
	str, ok := s.(string)
	if !ok {
		return -1, fmt.Errorf("rfind() requires string, got %T", s)
	}
	substr, ok := sub.(string)
	if !ok {
		return -1, fmt.Errorf("rfind() substring must be string, got %T", sub)
	}
	return int64(strings.LastIndex(str, substr)), nil
}

// StrPartition splits string into 3 parts at first occurrence of separator
func StrPartition(s, sep any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("partition() requires string, got %T", s)
	}
	separator, ok := sep.(string)
	if !ok {
		return nil, fmt.Errorf("partition() separator must be string, got %T", sep)
	}
	idx := strings.Index(str, separator)
	if idx == -1 {
		return []any{str, "", ""}, nil
	}
	return []any{str[:idx], separator, str[idx+len(separator):]}, nil
}

// StrRpartition splits string into 3 parts at last occurrence of separator
func StrRpartition(s, sep any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("rpartition() requires string, got %T", s)
	}
	separator, ok := sep.(string)
	if !ok {
		return nil, fmt.Errorf("rpartition() separator must be string, got %T", sep)
	}
	idx := strings.LastIndex(str, separator)
	if idx == -1 {
		return []any{"", "", str}, nil
	}
	return []any{str[:idx], separator, str[idx+len(separator):]}, nil
}

// StrExpandtabs replaces tabs with spaces
func StrExpandtabs(s any, tabsize any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("expandtabs() requires string, got %T", s)
	}
	size := int64(8)
	if tabsize != nil {
		if ts, ok := tabsize.(int64); ok {
			size = ts
		}
	}
	return strings.ReplaceAll(str, "\t", strings.Repeat(" ", int(size))), nil
}

// StrIsspace checks if string contains only whitespace
func StrIsspace(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_space() requires string, got %T", s)
	}
	if len(str) == 0 {
		return false, nil
	}
	for _, r := range str {
		if r != ' ' && r != '\t' && r != '\n' && r != '\r' && r != '\f' && r != '\v' {
			return false, nil
		}
	}
	return true, nil
}

// StrIsupper checks if all cased characters are uppercase
func StrIsupper(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_upper() requires string, got %T", s)
	}
	hasCased := false
	for _, r := range str {
		if r >= 'a' && r <= 'z' {
			return false, nil
		}
		if r >= 'A' && r <= 'Z' {
			hasCased = true
		}
	}
	return hasCased, nil
}

// StrIslower checks if all cased characters are lowercase
func StrIslower(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_lower() requires string, got %T", s)
	}
	hasCased := false
	for _, r := range str {
		if r >= 'A' && r <= 'Z' {
			return false, nil
		}
		if r >= 'a' && r <= 'z' {
			hasCased = true
		}
	}
	return hasCased, nil
}

// StrIstitle checks if string is titlecased
func StrIstitle(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_title() requires string, got %T", s)
	}
	if len(str) == 0 {
		return false, nil
	}
	prevCased := false
	hasCased := false
	for _, r := range str {
		if r >= 'A' && r <= 'Z' {
			if prevCased {
				return false, nil
			}
			prevCased = true
			hasCased = true
		} else if r >= 'a' && r <= 'z' {
			if !prevCased {
				return false, nil
			}
			prevCased = true
			hasCased = true
		} else {
			prevCased = false
		}
	}
	return hasCased, nil
}

// StrIsnumeric checks if string contains only numeric characters
func StrIsnumeric(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_numeric() requires string, got %T", s)
	}
	if len(str) == 0 {
		return false, nil
	}
	for _, r := range str {
		if r < '0' || r > '9' {
			return false, nil
		}
	}
	return true, nil
}

// ================== Text Processing Methods ==================

// StrExtractNumbers extracts all numbers from a string
func StrExtractNumbers(s any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("extract_numbers() requires string, got %T", s)
	}
	var result []any
	var current strings.Builder
	inNumber := false
	hasDecimal := false

	for i, r := range str {
		if r >= '0' && r <= '9' {
			current.WriteRune(r)
			inNumber = true
		} else if r == '.' && inNumber && !hasDecimal && i+1 < len(str) {
			next := rune(str[i+1])
			if next >= '0' && next <= '9' {
				current.WriteRune(r)
				hasDecimal = true
			} else if current.Len() > 0 {
				if num, err := parseNumber(current.String()); err == nil {
					result = append(result, num)
				}
				current.Reset()
				inNumber = false
				hasDecimal = false
			}
		} else if r == '-' && !inNumber && i+1 < len(str) {
			next := rune(str[i+1])
			if next >= '0' && next <= '9' {
				current.WriteRune(r)
			}
		} else if inNumber {
			if current.Len() > 0 {
				if num, err := parseNumber(current.String()); err == nil {
					result = append(result, num)
				}
			}
			current.Reset()
			inNumber = false
			hasDecimal = false
		}
	}
	if current.Len() > 0 {
		if num, err := parseNumber(current.String()); err == nil {
			result = append(result, num)
		}
	}
	return result, nil
}

func parseNumber(s string) (any, error) {
	if strings.Contains(s, ".") {
		return strconv.ParseFloat(s, 64)
	}
	return strconv.ParseInt(s, 10, 64)
}

// StrExtractEmails extracts email addresses from a string
func StrExtractEmails(s any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("extract_emails() requires string, got %T", s)
	}
	var result []any
	words := strings.Fields(str)
	for _, word := range words {
		// Simple email pattern: something@something.something
		word = strings.Trim(word, ".,;:!?<>()[]{}\"'")
		if idx := strings.Index(word, "@"); idx > 0 && idx < len(word)-1 {
			afterAt := word[idx+1:]
			if dotIdx := strings.Index(afterAt, "."); dotIdx > 0 && dotIdx < len(afterAt)-1 {
				result = append(result, word)
			}
		}
	}
	return result, nil
}

// StrExtractUrls extracts URLs from a string
func StrExtractUrls(s any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("extract_urls() requires string, got %T", s)
	}
	var result []any
	words := strings.Fields(str)
	for _, word := range words {
		word = strings.Trim(word, ".,;:!?<>()[]{}\"'")
		if strings.HasPrefix(word, "http://") || strings.HasPrefix(word, "https://") {
			result = append(result, word)
		} else if strings.HasPrefix(word, "www.") {
			result = append(result, "http://"+word)
		}
	}
	return result, nil
}

// StrExtractHashtags extracts hashtags from a string
func StrExtractHashtags(s any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("extract_hashtags() requires string, got %T", s)
	}
	var result []any
	words := strings.Fields(str)
	for _, word := range words {
		if strings.HasPrefix(word, "#") && len(word) > 1 {
			tag := strings.TrimLeft(word, "#")
			tag = strings.TrimRight(tag, ".,;:!?")
			if len(tag) > 0 {
				result = append(result, tag)
			}
		}
	}
	return result, nil
}

// StrExtractMentions extracts @mentions from a string
func StrExtractMentions(s any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("extract_mentions() requires string, got %T", s)
	}
	var result []any
	words := strings.Fields(str)
	for _, word := range words {
		if strings.HasPrefix(word, "@") && len(word) > 1 {
			mention := strings.TrimLeft(word, "@")
			mention = strings.TrimRight(mention, ".,;:!?")
			if len(mention) > 0 {
				result = append(result, mention)
			}
		}
	}
	return result, nil
}

// StrRemoveHtml strips HTML tags from a string
func StrRemoveHtml(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("remove_html() requires string, got %T", s)
	}
	var result strings.Builder
	inTag := false
	for _, r := range str {
		if r == '<' {
			inTag = true
		} else if r == '>' {
			inTag = false
		} else if !inTag {
			result.WriteRune(r)
		}
	}
	return result.String(), nil
}

// StrRemovePunctuation removes all punctuation from a string
func StrRemovePunctuation(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("remove_punctuation() requires string, got %T", s)
	}
	var result strings.Builder
	for _, r := range str {
		if (r >= 'a' && r <= 'z') || (r >= 'A' && r <= 'Z') || (r >= '0' && r <= '9') || r == ' ' || r == '\t' || r == '\n' {
			result.WriteRune(r)
		}
	}
	return result.String(), nil
}

// StrNormalizeWhitespace collapses all whitespace to single spaces
func StrNormalizeWhitespace(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("normalize_whitespace() requires string, got %T", s)
	}
	return strings.Join(strings.Fields(str), " "), nil
}

// StrToAscii removes non-ASCII characters
func StrToAscii(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("to_ascii() requires string, got %T", s)
	}
	var result strings.Builder
	for _, r := range str {
		if r < 128 {
			result.WriteRune(r)
		}
	}
	return result.String(), nil
}

// StrTransliterate converts accented characters to ASCII equivalents
func StrTransliterate(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("transliterate() requires string, got %T", s)
	}
	replacements := map[rune]string{
		'á': "a", 'à': "a", 'â': "a", 'ä': "a", 'ã': "a", 'å': "a",
		'é': "e", 'è': "e", 'ê': "e", 'ë': "e",
		'í': "i", 'ì': "i", 'î': "i", 'ï': "i",
		'ó': "o", 'ò': "o", 'ô': "o", 'ö': "o", 'õ': "o",
		'ú': "u", 'ù': "u", 'û': "u", 'ü': "u",
		'ñ': "n", 'ç': "c", 'ß': "ss",
		'Á': "A", 'À': "A", 'Â': "A", 'Ä': "A", 'Ã': "A", 'Å': "A",
		'É': "E", 'È': "E", 'Ê': "E", 'Ë': "E",
		'Í': "I", 'Ì': "I", 'Î': "I", 'Ï': "I",
		'Ó': "O", 'Ò': "O", 'Ô': "O", 'Ö': "O", 'Õ': "O",
		'Ú': "U", 'Ù': "U", 'Û': "U", 'Ü': "U",
		'Ñ': "N", 'Ç': "C",
	}
	var result strings.Builder
	for _, r := range str {
		if replacement, ok := replacements[r]; ok {
			result.WriteString(replacement)
		} else {
			result.WriteRune(r)
		}
	}
	return result.String(), nil
}

// ================== Case Conversion Methods ==================

// StrToSnakeCase converts string to snake_case
func StrToSnakeCase(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("to_snake_case() requires string, got %T", s)
	}
	var result strings.Builder
	for i, r := range str {
		if r >= 'A' && r <= 'Z' {
			if i > 0 {
				result.WriteRune('_')
			}
			result.WriteRune(r + 32)
		} else if r == ' ' || r == '-' {
			result.WriteRune('_')
		} else {
			result.WriteRune(r)
		}
	}
	// Clean up multiple underscores
	res := result.String()
	for strings.Contains(res, "__") {
		res = strings.ReplaceAll(res, "__", "_")
	}
	return strings.Trim(res, "_"), nil
}

// StrToCamelCase converts string to camelCase
func StrToCamelCase(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("to_camel_case() requires string, got %T", s)
	}
	words := splitIntoWords(str)
	if len(words) == 0 {
		return "", nil
	}
	var result strings.Builder
	result.WriteString(strings.ToLower(words[0]))
	for _, word := range words[1:] {
		if len(word) > 0 {
			result.WriteString(strings.ToUpper(string(word[0])))
			result.WriteString(strings.ToLower(word[1:]))
		}
	}
	return result.String(), nil
}

// StrToPascalCase converts string to PascalCase
func StrToPascalCase(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("to_pascal_case() requires string, got %T", s)
	}
	words := splitIntoWords(str)
	var result strings.Builder
	for _, word := range words {
		if len(word) > 0 {
			result.WriteString(strings.ToUpper(string(word[0])))
			result.WriteString(strings.ToLower(word[1:]))
		}
	}
	return result.String(), nil
}

// StrToKebabCase converts string to kebab-case
func StrToKebabCase(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("to_kebab_case() requires string, got %T", s)
	}
	var result strings.Builder
	for i, r := range str {
		if r >= 'A' && r <= 'Z' {
			if i > 0 {
				result.WriteRune('-')
			}
			result.WriteRune(r + 32)
		} else if r == ' ' || r == '_' {
			result.WriteRune('-')
		} else {
			result.WriteRune(r)
		}
	}
	res := result.String()
	for strings.Contains(res, "--") {
		res = strings.ReplaceAll(res, "--", "-")
	}
	return strings.Trim(res, "-"), nil
}

// StrToTitleCase converts string to Smart Title Case
func StrToTitleCase(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("to_title_case() requires string, got %T", s)
	}
	// Words to keep lowercase (unless first word)
	minor := map[string]bool{
		"a": true, "an": true, "the": true, "and": true, "but": true,
		"or": true, "for": true, "nor": true, "on": true, "at": true,
		"to": true, "by": true, "of": true, "in": true, "as": true,
	}
	words := strings.Fields(str)
	for i, word := range words {
		lower := strings.ToLower(word)
		if i == 0 || !minor[lower] {
			if len(word) > 0 {
				words[i] = strings.ToUpper(string(word[0])) + strings.ToLower(word[1:])
			}
		} else {
			words[i] = lower
		}
	}
	return strings.Join(words, " "), nil
}

// StrToSentenceCase capitalizes first letter of each sentence
func StrToSentenceCase(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("to_sentence_case() requires string, got %T", s)
	}
	str = strings.ToLower(str)
	var result strings.Builder
	capitalizeNext := true
	for _, r := range str {
		if capitalizeNext && r >= 'a' && r <= 'z' {
			result.WriteRune(r - 32)
			capitalizeNext = false
		} else {
			result.WriteRune(r)
			if r == '.' || r == '!' || r == '?' {
				capitalizeNext = true
			}
		}
	}
	return result.String(), nil
}

// StrHumanize converts variable names to human readable form
func StrHumanize(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("humanize() requires string, got %T", s)
	}
	// Replace underscores and hyphens with spaces
	str = strings.ReplaceAll(str, "_", " ")
	str = strings.ReplaceAll(str, "-", " ")
	// Add space before capital letters
	var result strings.Builder
	for i, r := range str {
		if r >= 'A' && r <= 'Z' && i > 0 {
			prev := rune(str[i-1])
			if prev >= 'a' && prev <= 'z' {
				result.WriteRune(' ')
			}
		}
		result.WriteRune(r)
	}
	// Capitalize first letter, lowercase rest
	res := result.String()
	if len(res) > 0 {
		return strings.ToUpper(string(res[0])) + strings.ToLower(res[1:]), nil
	}
	return "", nil
}

// StrDehumanize converts human readable form to snake_case
func StrDehumanize(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("dehumanize() requires string, got %T", s)
	}
	return StrToSnakeCase(strings.ToLower(str))
}

func splitIntoWords(s string) []string {
	// Split on spaces, underscores, hyphens, and camelCase boundaries
	var words []string
	var current strings.Builder
	for i, r := range s {
		if r == ' ' || r == '_' || r == '-' {
			if current.Len() > 0 {
				words = append(words, current.String())
				current.Reset()
			}
		} else if r >= 'A' && r <= 'Z' && i > 0 {
			prev := rune(s[i-1])
			if prev >= 'a' && prev <= 'z' {
				if current.Len() > 0 {
					words = append(words, current.String())
					current.Reset()
				}
			}
			current.WriteRune(r)
		} else {
			current.WriteRune(r)
		}
	}
	if current.Len() > 0 {
		words = append(words, current.String())
	}
	return words
}

// ================== Validation Methods ==================

// StrIsEmail checks if string is a valid email format
func StrIsEmail(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_email() requires string, got %T", s)
	}
	if len(str) < 5 {
		return false, nil
	}
	atIdx := strings.Index(str, "@")
	if atIdx < 1 || atIdx >= len(str)-3 {
		return false, nil
	}
	afterAt := str[atIdx+1:]
	dotIdx := strings.LastIndex(afterAt, ".")
	if dotIdx < 1 || dotIdx >= len(afterAt)-1 {
		return false, nil
	}
	return true, nil
}

// StrIsUrl checks if string is a valid URL format
func StrIsUrl(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_url() requires string, got %T", s)
	}
	if strings.HasPrefix(str, "http://") || strings.HasPrefix(str, "https://") {
		afterProtocol := strings.TrimPrefix(strings.TrimPrefix(str, "https://"), "http://")
		if len(afterProtocol) > 0 && strings.Contains(afterProtocol, ".") {
			return true, nil
		}
	}
	return false, nil
}

// StrIsIp checks if string is a valid IP address (IPv4 or IPv6)
func StrIsIp(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_ip() requires string, got %T", s)
	}
	// Check IPv4
	parts := strings.Split(str, ".")
	if len(parts) == 4 {
		valid := true
		for _, part := range parts {
			n, err := strconv.Atoi(part)
			if err != nil || n < 0 || n > 255 {
				valid = false
				break
			}
		}
		if valid {
			return true, nil
		}
	}
	// Check IPv6 (simplified check)
	if strings.Contains(str, ":") && !strings.Contains(str, ".") {
		parts := strings.Split(str, ":")
		if len(parts) >= 3 && len(parts) <= 8 {
			return true, nil
		}
	}
	return false, nil
}

// StrIsUuid checks if string is a valid UUID format
func StrIsUuid(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_uuid() requires string, got %T", s)
	}
	// UUID format: xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx
	if len(str) != 36 {
		return false, nil
	}
	if str[8] != '-' || str[13] != '-' || str[18] != '-' || str[23] != '-' {
		return false, nil
	}
	hexParts := []string{str[0:8], str[9:13], str[14:18], str[19:23], str[24:36]}
	for _, part := range hexParts {
		for _, r := range part {
			if !((r >= '0' && r <= '9') || (r >= 'a' && r <= 'f') || (r >= 'A' && r <= 'F')) {
				return false, nil
			}
		}
	}
	return true, nil
}

// StrIsJson checks if string is valid JSON
func StrIsJson(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_json() requires string, got %T", s)
	}
	str = strings.TrimSpace(str)
	if len(str) == 0 {
		return false, nil
	}
	// Simple check: must start with { or [ and end with } or ]
	if (str[0] == '{' && str[len(str)-1] == '}') || (str[0] == '[' && str[len(str)-1] == ']') {
		// Count braces to ensure they're balanced
		braces := 0
		brackets := 0
		inString := false
		escaped := false
		for _, r := range str {
			if escaped {
				escaped = false
				continue
			}
			if r == '\\' && inString {
				escaped = true
				continue
			}
			if r == '"' {
				inString = !inString
			} else if !inString {
				switch r {
				case '{':
					braces++
				case '}':
					braces--
				case '[':
					brackets++
				case ']':
					brackets--
				}
			}
		}
		return braces == 0 && brackets == 0, nil
	}
	return false, nil
}

// StrIsHex checks if string contains only hexadecimal characters
func StrIsHex(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_hex() requires string, got %T", s)
	}
	if len(str) == 0 {
		return false, nil
	}
	// Skip 0x prefix if present
	if strings.HasPrefix(str, "0x") || strings.HasPrefix(str, "0X") {
		str = str[2:]
	}
	for _, r := range str {
		if !((r >= '0' && r <= '9') || (r >= 'a' && r <= 'f') || (r >= 'A' && r <= 'F')) {
			return false, nil
		}
	}
	return true, nil
}

// StrIsBase64 checks if string is valid base64
func StrIsBase64(s any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("is_base64() requires string, got %T", s)
	}
	if len(str) == 0 || len(str)%4 != 0 {
		return false, nil
	}
	for i, r := range str {
		if r == '=' {
			// Padding only allowed at end
			if i < len(str)-2 {
				return false, nil
			}
		} else if !((r >= 'A' && r <= 'Z') || (r >= 'a' && r <= 'z') || (r >= '0' && r <= '9') || r == '+' || r == '/') {
			return false, nil
		}
	}
	return true, nil
}

// StrMatches checks if string matches a simple pattern (* wildcards)
func StrMatches(s, pattern any) (bool, error) {
	str, ok := s.(string)
	if !ok {
		return false, fmt.Errorf("matches() requires string, got %T", s)
	}
	pat, ok := pattern.(string)
	if !ok {
		return false, fmt.Errorf("matches() pattern must be string, got %T", pattern)
	}
	return matchWildcard(pat, str), nil
}

func matchWildcard(pattern, str string) bool {
	if pattern == "*" {
		return true
	}
	if pattern == "" {
		return str == ""
	}
	if str == "" {
		return pattern == "*" || pattern == ""
	}
	if pattern[0] == '*' {
		return matchWildcard(pattern[1:], str) || matchWildcard(pattern, str[1:])
	}
	if pattern[0] == str[0] || pattern[0] == '?' {
		return matchWildcard(pattern[1:], str[1:])
	}
	return false
}

// ================== Transformation Methods ==================

// StrRot13 applies ROT13 cipher
func StrRot13(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("rot13() requires string, got %T", s)
	}
	var result strings.Builder
	for _, r := range str {
		if r >= 'a' && r <= 'z' {
			result.WriteRune('a' + (r-'a'+13)%26)
		} else if r >= 'A' && r <= 'Z' {
			result.WriteRune('A' + (r-'A'+13)%26)
		} else {
			result.WriteRune(r)
		}
	}
	return result.String(), nil
}

// StrReverseWords reverses word order in a string
func StrReverseWords(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("reverse_words() requires string, got %T", s)
	}
	words := strings.Fields(str)
	for i, j := 0, len(words)-1; i < j; i, j = i+1, j-1 {
		words[i], words[j] = words[j], words[i]
	}
	return strings.Join(words, " "), nil
}

// StrShuffle randomly shuffles characters in a string
func StrShuffle(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("shuffle() requires string, got %T", s)
	}
	runes := []rune(str)
	// Simple shuffle using current time as seed
	seed := time.Now().UnixNano()
	for i := len(runes) - 1; i > 0; i-- {
		seed = seed*1103515245 + 12345
		j := int((seed / 65536) % int64(i+1))
		if j < 0 {
			j = -j
		}
		runes[i], runes[j] = runes[j], runes[i]
	}
	return string(runes), nil
}

// StrSortChars sorts characters in a string
func StrSortChars(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("sort_chars() requires string, got %T", s)
	}
	runes := []rune(str)
	// Simple bubble sort
	for i := 0; i < len(runes)-1; i++ {
		for j := 0; j < len(runes)-1-i; j++ {
			if runes[j] > runes[j+1] {
				runes[j], runes[j+1] = runes[j+1], runes[j]
			}
		}
	}
	return string(runes), nil
}

// StrUniqueChars removes duplicate characters
func StrUniqueChars(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("unique_chars() requires string, got %T", s)
	}
	seen := make(map[rune]bool)
	var result strings.Builder
	for _, r := range str {
		if !seen[r] {
			seen[r] = true
			result.WriteRune(r)
		}
	}
	return result.String(), nil
}

// StrCharFrequency returns character frequency as a dict
func StrCharFrequency(s any) (map[string]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("char_frequency() requires string, got %T", s)
	}
	result := make(map[string]any)
	for _, r := range str {
		key := string(r)
		if count, exists := result[key]; exists {
			result[key] = count.(int64) + 1
		} else {
			result[key] = int64(1)
		}
	}
	return result, nil
}

// ================== Linguistic Methods ==================

// StrWordCount counts words in a string
func StrWordCount(s any) (int64, error) {
	str, ok := s.(string)
	if !ok {
		return 0, fmt.Errorf("word_count() requires string, got %T", s)
	}
	return int64(len(strings.Fields(str))), nil
}

// StrReadingTime estimates reading time in minutes
func StrReadingTime(s any, wpm any) (float64, error) {
	str, ok := s.(string)
	if !ok {
		return 0, fmt.Errorf("reading_time() requires string, got %T", s)
	}
	wordsPerMin := int64(200)
	if wpm != nil {
		if w, ok := wpm.(int64); ok {
			wordsPerMin = w
		}
	}
	wordCount := int64(len(strings.Fields(str)))
	return float64(wordCount) / float64(wordsPerMin), nil
}

// StrLevenshtein calculates Levenshtein edit distance
func StrLevenshtein(s, t any) (int64, error) {
	str1, ok := s.(string)
	if !ok {
		return 0, fmt.Errorf("levenshtein() requires string, got %T", s)
	}
	str2, ok := t.(string)
	if !ok {
		return 0, fmt.Errorf("levenshtein() second arg must be string, got %T", t)
	}
	if str1 == str2 {
		return 0, nil
	}
	if len(str1) == 0 {
		return int64(len(str2)), nil
	}
	if len(str2) == 0 {
		return int64(len(str1)), nil
	}
	r1, r2 := []rune(str1), []rune(str2)
	// Create distance matrix
	prev := make([]int, len(r2)+1)
	curr := make([]int, len(r2)+1)
	for j := range prev {
		prev[j] = j
	}
	for i := 1; i <= len(r1); i++ {
		curr[0] = i
		for j := 1; j <= len(r2); j++ {
			cost := 0
			if r1[i-1] != r2[j-1] {
				cost = 1
			}
			curr[j] = min(min(prev[j]+1, curr[j-1]+1), prev[j-1]+cost)
		}
		prev, curr = curr, prev
	}
	return int64(prev[len(r2)]), nil
}

func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}

// StrSimilarity calculates similarity ratio (0-1)
func StrSimilarity(s, t any) (float64, error) {
	str1, ok := s.(string)
	if !ok {
		return 0, fmt.Errorf("similarity() requires string, got %T", s)
	}
	str2, ok := t.(string)
	if !ok {
		return 0, fmt.Errorf("similarity() second arg must be string, got %T", t)
	}
	if str1 == str2 {
		return 1.0, nil
	}
	dist, _ := StrLevenshtein(s, t)
	maxLen := len(str1)
	if len(str2) > maxLen {
		maxLen = len(str2)
	}
	if maxLen == 0 {
		return 1.0, nil
	}
	return 1.0 - float64(dist)/float64(maxLen), nil
}

// StrSoundex returns Soundex phonetic code
func StrSoundex(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("soundex() requires string, got %T", s)
	}
	if len(str) == 0 {
		return "", nil
	}
	str = strings.ToUpper(str)
	// Keep first letter
	result := string(str[0])
	// Soundex codes
	codes := map[rune]rune{
		'B': '1', 'F': '1', 'P': '1', 'V': '1',
		'C': '2', 'G': '2', 'J': '2', 'K': '2', 'Q': '2', 'S': '2', 'X': '2', 'Z': '2',
		'D': '3', 'T': '3',
		'L': '4',
		'M': '5', 'N': '5',
		'R': '6',
	}
	prevCode := codes[rune(str[0])]
	for _, r := range str[1:] {
		if code, ok := codes[r]; ok && code != prevCode {
			result += string(code)
			prevCode = code
			if len(result) == 4 {
				break
			}
		} else if r == 'A' || r == 'E' || r == 'I' || r == 'O' || r == 'U' || r == 'H' || r == 'W' || r == 'Y' {
			prevCode = 0
		}
	}
	// Pad with zeros
	for len(result) < 4 {
		result += "0"
	}
	return result, nil
}

// ================== Utility Methods ==================

// StrChunk splits string into chunks of specified size
func StrChunk(s any, size any) ([]any, error) {
	str, ok := s.(string)
	if !ok {
		return nil, fmt.Errorf("chunk() requires string, got %T", s)
	}
	sz, ok := size.(int64)
	if !ok {
		return nil, fmt.Errorf("chunk() size must be int, got %T", size)
	}
	if sz <= 0 {
		return nil, fmt.Errorf("chunk() size must be positive")
	}
	var result []any
	runes := []rune(str)
	for i := 0; i < len(runes); i += int(sz) {
		end := i + int(sz)
		if end > len(runes) {
			end = len(runes)
		}
		result = append(result, string(runes[i:end]))
	}
	return result, nil
}

// StrFirst returns first n characters
func StrFirst(s any, n any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("first() requires string, got %T", s)
	}
	count, ok := n.(int64)
	if !ok {
		return "", fmt.Errorf("first() count must be int, got %T", n)
	}
	runes := []rune(str)
	if count <= 0 {
		return "", nil
	}
	if int(count) >= len(runes) {
		return str, nil
	}
	return string(runes[:count]), nil
}

// StrLast returns last n characters
func StrLast(s any, n any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("last() requires string, got %T", s)
	}
	count, ok := n.(int64)
	if !ok {
		return "", fmt.Errorf("last() count must be int, got %T", n)
	}
	runes := []rune(str)
	if count <= 0 {
		return "", nil
	}
	if int(count) >= len(runes) {
		return str, nil
	}
	return string(runes[len(runes)-int(count):]), nil
}

// StrAt returns character at index (supports negative indexing)
func StrAt(s any, idx any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("at() requires string, got %T", s)
	}
	index, ok := idx.(int64)
	if !ok {
		return "", fmt.Errorf("at() index must be int, got %T", idx)
	}
	runes := []rune(str)
	if index < 0 {
		index = int64(len(runes)) + index
	}
	if index < 0 || int(index) >= len(runes) {
		return "", nil
	}
	return string(runes[index]), nil
}

// StrSafeSlice slices string without errors (returns "" if out of bounds)
func StrSafeSlice(s any, start any, end any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("safe_slice() requires string, got %T", s)
	}
	startIdx, ok := start.(int64)
	if !ok {
		return "", fmt.Errorf("safe_slice() start must be int, got %T", start)
	}
	endIdx, ok := end.(int64)
	if !ok {
		return "", fmt.Errorf("safe_slice() end must be int, got %T", end)
	}
	runes := []rune(str)
	length := int64(len(runes))
	if startIdx < 0 {
		startIdx = 0
	}
	if endIdx > length {
		endIdx = length
	}
	if startIdx >= endIdx || startIdx >= length {
		return "", nil
	}
	return string(runes[startIdx:endIdx]), nil
}

// StrInsert inserts string at specified position
func StrInsert(s any, idx any, insert any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("insert() requires string, got %T", s)
	}
	index, ok := idx.(int64)
	if !ok {
		return "", fmt.Errorf("insert() index must be int, got %T", idx)
	}
	ins, ok := insert.(string)
	if !ok {
		return "", fmt.Errorf("insert() insert must be string, got %T", insert)
	}
	runes := []rune(str)
	if index < 0 {
		index = 0
	}
	if int(index) >= len(runes) {
		return str + ins, nil
	}
	return string(runes[:index]) + ins + string(runes[index:]), nil
}

// StrDelete deletes characters from start to end
func StrDelete(s any, start any, end any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("delete() requires string, got %T", s)
	}
	startIdx, ok := start.(int64)
	if !ok {
		return "", fmt.Errorf("delete() start must be int, got %T", start)
	}
	endIdx, ok := end.(int64)
	if !ok {
		return "", fmt.Errorf("delete() end must be int, got %T", end)
	}
	runes := []rune(str)
	if startIdx < 0 {
		startIdx = 0
	}
	if endIdx > int64(len(runes)) {
		endIdx = int64(len(runes))
	}
	if startIdx >= endIdx {
		return str, nil
	}
	return string(runes[:startIdx]) + string(runes[endIdx:]), nil
}

// StrOverlay overlays a string at specified position
func StrOverlay(s any, overlay any, start any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("overlay() requires string, got %T", s)
	}
	ovl, ok := overlay.(string)
	if !ok {
		return "", fmt.Errorf("overlay() overlay must be string, got %T", overlay)
	}
	startIdx, ok := start.(int64)
	if !ok {
		return "", fmt.Errorf("overlay() start must be int, got %T", start)
	}
	runes := []rune(str)
	ovlRunes := []rune(ovl)
	if startIdx < 0 {
		startIdx = 0
	}
	endIdx := startIdx + int64(len(ovlRunes))
	if int(startIdx) >= len(runes) {
		return str + ovl, nil
	}
	result := string(runes[:startIdx]) + ovl
	if int(endIdx) < len(runes) {
		result += string(runes[endIdx:])
	}
	return result, nil
}

// ================== Security Methods ==================

// StrRedact redacts sensitive patterns from a string
func StrRedact(s any, pattern any, replacement any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("redact() requires string, got %T", s)
	}
	pat, ok := pattern.(string)
	if !ok {
		return "", fmt.Errorf("redact() pattern must be string, got %T", pattern)
	}
	repl := "[REDACTED]"
	if replacement != nil {
		if r, ok := replacement.(string); ok {
			repl = r
		}
	}
	return strings.ReplaceAll(str, pat, repl), nil
}

// StrObfuscate performs simple reversible obfuscation
func StrObfuscate(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("obfuscate() requires string, got %T", s)
	}
	// Simple XOR obfuscation with a fixed key
	key := byte(42)
	result := make([]byte, len(str))
	for i := 0; i < len(str); i++ {
		result[i] = str[i] ^ key
	}
	// Convert to hex for safe string representation
	var hex strings.Builder
	for _, b := range result {
		hex.WriteString(fmt.Sprintf("%02x", b))
	}
	return hex.String(), nil
}

// StrDeobfuscate reverses the obfuscation
func StrDeobfuscate(s any) (string, error) {
	str, ok := s.(string)
	if !ok {
		return "", fmt.Errorf("deobfuscate() requires string, got %T", s)
	}
	if len(str)%2 != 0 {
		return "", fmt.Errorf("invalid obfuscated string")
	}
	key := byte(42)
	var result strings.Builder
	for i := 0; i < len(str); i += 2 {
		b, err := strconv.ParseUint(str[i:i+2], 16, 8)
		if err != nil {
			return "", fmt.Errorf("invalid obfuscated string")
		}
		result.WriteByte(byte(b) ^ key)
	}
	return result.String(), nil
}

// ================== File Operations ==================

// File represents an open file
type File struct {
	handle *os.File
	mode   string
}

// OpenFile opens a file with the specified mode
func OpenFile(path string, mode string) (*File, error) {
	var flag int
	switch mode {
	case "r", "":
		flag = os.O_RDONLY
	case "w":
		flag = os.O_WRONLY | os.O_CREATE | os.O_TRUNC
	case "a":
		flag = os.O_WRONLY | os.O_CREATE | os.O_APPEND
	case "r+":
		flag = os.O_RDWR
	case "w+":
		flag = os.O_RDWR | os.O_CREATE | os.O_TRUNC
	case "a+":
		flag = os.O_RDWR | os.O_CREATE | os.O_APPEND
	default:
		return nil, fmt.Errorf("invalid file mode: %s", mode)
	}

	handle, err := os.OpenFile(path, flag, 0644)
	if err != nil {
		return nil, err
	}

	return &File{handle: handle, mode: mode}, nil
}

// Read reads the entire file content
func (f *File) Read() (string, error) {
	content, err := os.ReadFile(f.handle.Name())
	if err != nil {
		return "", err
	}
	return string(content), nil
}

// ReadLines reads all lines from the file
func (f *File) ReadLines() ([]any, error) {
	content, err := f.Read()
	if err != nil {
		return nil, err
	}
	lines := strings.Split(content, "\n")
	result := make([]any, len(lines))
	for i, line := range lines {
		result[i] = line
	}
	return result, nil
}

// Write writes content to the file
func (f *File) Write(content string) error {
	_, err := f.handle.WriteString(content)
	return err
}

// WriteLine writes a line to the file
func (f *File) WriteLine(content string) error {
	_, err := f.handle.WriteString(content + "\n")
	return err
}

// Close closes the file
func (f *File) Close() error {
	return f.handle.Close()
}

// ================== CSV Operations ==================

// CSVReader reads CSV files
type CSVReader struct {
	file       *File
	delimiter  string
	headers    []string
	rows       [][]string
	lines      []string
	currentRow int
	loaded     bool
	inferTypes bool
}

// NewCSVReader creates a new CSV reader
func NewCSVReader(path string) (*CSVReader, error) {
	file, err := OpenFile(path, "r")
	if err != nil {
		return nil, err
	}
	return &CSVReader{
		file:       file,
		delimiter:  ",",
		currentRow: 0,
		inferTypes: false,
	}, nil
}

// SetDelimiter sets the CSV delimiter
func (c *CSVReader) SetDelimiter(delim string) {
	c.delimiter = delim
}

// SetInferTypes enables automatic type conversion for numbers
func (c *CSVReader) SetInferTypes(infer bool) {
	c.inferTypes = infer
}

// loadFile reads and parses the file content
func (c *CSVReader) loadFile() error {
	if c.loaded {
		return nil
	}
	content, err := c.file.Read()
	if err != nil {
		return err
	}
	c.file.Close()
	c.lines = strings.Split(strings.TrimSpace(content), "\n")
	if len(c.lines) > 0 {
		c.headers = c.parseLine(c.lines[0])
	}
	c.currentRow = 1 // Start after headers
	c.loaded = true
	return nil
}

// inferValue attempts to convert string to int or float
func (c *CSVReader) inferValue(val string) any {
	if !c.inferTypes {
		return val
	}
	// Try int first
	if i, err := strconv.ParseInt(val, 10, 64); err == nil {
		return i
	}
	// Try float
	if f, err := strconv.ParseFloat(val, 64); err == nil {
		return f
	}
	// Try bool
	lower := strings.ToLower(val)
	if lower == "true" {
		return true
	}
	if lower == "false" {
		return false
	}
	return val
}

// ReadAll reads all CSV data as list of dictionaries
func (c *CSVReader) ReadAll() ([]map[string]any, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	var result []map[string]any
	for i := 1; i < len(c.lines); i++ {
		if strings.TrimSpace(c.lines[i]) == "" {
			continue
		}
		values := c.parseLine(c.lines[i])
		row := make(map[string]any)
		for j, header := range c.headers {
			if j < len(values) {
				row[header] = c.inferValue(values[j])
			} else {
				row[header] = ""
			}
		}
		result = append(result, row)
	}
	c.rows = nil // Clear raw storage
	return result, nil
}

// ReadAllRaw reads all CSV data as list of lists (no headers)
func (c *CSVReader) ReadAllRaw() ([][]any, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	var result [][]any
	for i := 0; i < len(c.lines); i++ {
		if strings.TrimSpace(c.lines[i]) == "" {
			continue
		}
		values := c.parseLine(c.lines[i])
		row := make([]any, len(values))
		for j, v := range values {
			row[j] = c.inferValue(v)
		}
		result = append(result, row)
	}
	return result, nil
}

// ReadRow reads the next row as a dictionary (for streaming)
func (c *CSVReader) ReadRow() (map[string]any, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	// Skip empty lines
	for c.currentRow < len(c.lines) && strings.TrimSpace(c.lines[c.currentRow]) == "" {
		c.currentRow++
	}

	if c.currentRow >= len(c.lines) {
		return nil, nil // End of file
	}

	values := c.parseLine(c.lines[c.currentRow])
	c.currentRow++

	row := make(map[string]any)
	for j, header := range c.headers {
		if j < len(values) {
			row[header] = c.inferValue(values[j])
		} else {
			row[header] = ""
		}
	}
	return row, nil
}

// HasNext returns true if there are more rows to read
func (c *CSVReader) HasNext() bool {
	if !c.loaded {
		c.loadFile()
	}
	// Skip empty lines when checking
	tempRow := c.currentRow
	for tempRow < len(c.lines) && strings.TrimSpace(c.lines[tempRow]) == "" {
		tempRow++
	}
	return tempRow < len(c.lines)
}

// Reset resets the reader to the beginning (after headers)
func (c *CSVReader) Reset() {
	c.currentRow = 1
}

// RowCount returns the number of data rows (excluding header)
func (c *CSVReader) RowCount() (int, error) {
	if err := c.loadFile(); err != nil {
		return 0, err
	}
	count := 0
	for i := 1; i < len(c.lines); i++ {
		if strings.TrimSpace(c.lines[i]) != "" {
			count++
		}
	}
	return count, nil
}

// Headers returns the CSV headers
func (c *CSVReader) Headers() []any {
	if !c.loaded {
		c.loadFile()
	}
	result := make([]any, len(c.headers))
	for i, h := range c.headers {
		result[i] = h
	}
	return result
}

func (c *CSVReader) parseLine(line string) []string {
	var result []string
	var current strings.Builder
	inQuotes := false
	delimRune := rune(c.delimiter[0])

	runes := []rune(line)
	for i := 0; i < len(runes); i++ {
		ch := runes[i]
		switch {
		case ch == '"':
			// Handle escaped quotes ("")
			if inQuotes && i+1 < len(runes) && runes[i+1] == '"' {
				current.WriteRune('"')
				i++ // Skip next quote
			} else {
				inQuotes = !inQuotes
			}
		case ch == delimRune && !inQuotes:
			result = append(result, strings.TrimSpace(current.String()))
			current.Reset()
		default:
			current.WriteRune(ch)
		}
	}
	result = append(result, strings.TrimSpace(current.String()))
	return result
}

// ================== CSV Analytics ==================

// getColumnValues extracts all values for a specific column
func (c *CSVReader) getColumnValues(column string) ([]string, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	colIndex := -1
	for i, h := range c.headers {
		if h == column {
			colIndex = i
			break
		}
	}
	if colIndex == -1 {
		return nil, fmt.Errorf("column '%s' not found", column)
	}

	var values []string
	for i := 1; i < len(c.lines); i++ {
		if strings.TrimSpace(c.lines[i]) == "" {
			continue
		}
		row := c.parseLine(c.lines[i])
		if colIndex < len(row) {
			values = append(values, row[colIndex])
		}
	}
	return values, nil
}

// getNumericValues extracts numeric values from a column
func (c *CSVReader) getNumericValues(column string) ([]float64, error) {
	strValues, err := c.getColumnValues(column)
	if err != nil {
		return nil, err
	}

	var nums []float64
	for _, v := range strValues {
		if f, err := strconv.ParseFloat(v, 64); err == nil {
			nums = append(nums, f)
		}
	}
	return nums, nil
}

// Column returns all values for a specific column
func (c *CSVReader) Column(name string) ([]any, error) {
	vals, err := c.getColumnValues(name)
	if err != nil {
		return nil, err
	}
	result := make([]any, len(vals))
	for i, v := range vals {
		result[i] = c.inferValue(v)
	}
	return result, nil
}

// ColumnStats holds statistics for a numeric column
type ColumnStats struct {
	Count  int
	Min    float64
	Max    float64
	Sum    float64
	Mean   float64
	Median float64
	StdDev float64
	Nulls  int
	Unique int
}

// Stats returns statistics for a numeric column
func (c *CSVReader) Stats(column string) (map[string]any, error) {
	strValues, err := c.getColumnValues(column)
	if err != nil {
		return nil, err
	}

	var nums []float64
	nullCount := 0
	uniqueMap := make(map[string]bool)

	for _, v := range strValues {
		uniqueMap[v] = true
		if v == "" {
			nullCount++
			continue
		}
		if f, err := strconv.ParseFloat(v, 64); err == nil {
			nums = append(nums, f)
		}
	}

	result := make(map[string]any)
	result["count"] = len(strValues)
	result["nulls"] = nullCount
	result["unique"] = len(uniqueMap)

	if len(nums) == 0 {
		result["numeric"] = false
		return result, nil
	}

	result["numeric"] = true

	// Calculate stats
	sum := 0.0
	min := nums[0]
	max := nums[0]
	for _, n := range nums {
		sum += n
		if n < min {
			min = n
		}
		if n > max {
			max = n
		}
	}
	mean := sum / float64(len(nums))

	// Calculate median
	sorted := make([]float64, len(nums))
	copy(sorted, nums)
	sortFloats(sorted)
	var median float64
	if len(sorted)%2 == 0 {
		median = (sorted[len(sorted)/2-1] + sorted[len(sorted)/2]) / 2
	} else {
		median = sorted[len(sorted)/2]
	}

	// Calculate standard deviation
	variance := 0.0
	for _, n := range nums {
		variance += (n - mean) * (n - mean)
	}
	stdDev := math.Sqrt(variance / float64(len(nums)))

	result["min"] = min
	result["max"] = max
	result["sum"] = sum
	result["mean"] = mean
	result["median"] = median
	result["stddev"] = stdDev

	return result, nil
}

// sortFloats sorts a slice of floats in place
func sortFloats(nums []float64) {
	for i := 0; i < len(nums)-1; i++ {
		for j := i + 1; j < len(nums); j++ {
			if nums[j] < nums[i] {
				nums[i], nums[j] = nums[j], nums[i]
			}
		}
	}
}

// Describe returns a summary of all columns (like pandas describe)
func (c *CSVReader) Describe() (map[string]map[string]any, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	result := make(map[string]map[string]any)
	for _, header := range c.headers {
		stats, err := c.Stats(header)
		if err != nil {
			continue
		}
		result[header] = stats
	}
	return result, nil
}

// UniqueValues returns all unique values in a column
func (c *CSVReader) UniqueValues(column string) ([]any, error) {
	vals, err := c.getColumnValues(column)
	if err != nil {
		return nil, err
	}

	seen := make(map[string]bool)
	var result []any
	for _, v := range vals {
		if !seen[v] {
			seen[v] = true
			result = append(result, c.inferValue(v))
		}
	}
	return result, nil
}

// ValueCounts returns count of each unique value in a column
func (c *CSVReader) ValueCounts(column string) (map[string]int, error) {
	vals, err := c.getColumnValues(column)
	if err != nil {
		return nil, err
	}

	counts := make(map[string]int)
	for _, v := range vals {
		counts[v]++
	}
	return counts, nil
}

// Top returns the N most common values in a column
func (c *CSVReader) Top(column string, n int) ([]map[string]any, error) {
	counts, err := c.ValueCounts(column)
	if err != nil {
		return nil, err
	}

	// Convert to slice for sorting
	type kv struct {
		Key   string
		Value int
	}
	var sorted []kv
	for k, v := range counts {
		sorted = append(sorted, kv{k, v})
	}

	// Sort by count descending
	for i := 0; i < len(sorted)-1; i++ {
		for j := i + 1; j < len(sorted); j++ {
			if sorted[j].Value > sorted[i].Value {
				sorted[i], sorted[j] = sorted[j], sorted[i]
			}
		}
	}

	// Take top N
	if n > len(sorted) {
		n = len(sorted)
	}

	result := make([]map[string]any, n)
	for i := 0; i < n; i++ {
		result[i] = map[string]any{
			"value": c.inferValue(sorted[i].Key),
			"count": sorted[i].Value,
		}
	}
	return result, nil
}

// NullCount returns the count of empty/null values per column
func (c *CSVReader) NullCount() (map[string]int, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	result := make(map[string]int)
	for _, header := range c.headers {
		vals, _ := c.getColumnValues(header)
		count := 0
		for _, v := range vals {
			if v == "" {
				count++
			}
		}
		result[header] = count
	}
	return result, nil
}

// DataTypes returns the inferred type for each column
func (c *CSVReader) DataTypes() (map[string]string, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	result := make(map[string]string)
	for _, header := range c.headers {
		vals, _ := c.getColumnValues(header)
		result[header] = inferColumnType(vals)
	}
	return result, nil
}

// inferColumnType determines the predominant type of a column
func inferColumnType(vals []string) string {
	if len(vals) == 0 {
		return "empty"
	}

	intCount, floatCount, boolCount, strCount := 0, 0, 0, 0
	for _, v := range vals {
		if v == "" {
			continue
		}
		if _, err := strconv.ParseInt(v, 10, 64); err == nil {
			intCount++
		} else if _, err := strconv.ParseFloat(v, 64); err == nil {
			floatCount++
		} else if strings.ToLower(v) == "true" || strings.ToLower(v) == "false" {
			boolCount++
		} else {
			strCount++
		}
	}

	// Determine predominant type
	max := intCount
	dtype := "int"
	if floatCount > max {
		max = floatCount
		dtype = "float"
	}
	if boolCount > max {
		max = boolCount
		dtype = "bool"
	}
	if strCount > max {
		dtype = "string"
	}
	if max == 0 {
		return "empty"
	}
	return dtype
}

// Correlate returns Pearson correlation between two numeric columns
func (c *CSVReader) Correlate(col1, col2 string) (float64, error) {
	vals1, err := c.getNumericValues(col1)
	if err != nil {
		return 0, err
	}
	vals2, err := c.getNumericValues(col2)
	if err != nil {
		return 0, err
	}

	// Need same length
	n := len(vals1)
	if len(vals2) < n {
		n = len(vals2)
	}
	if n == 0 {
		return 0, fmt.Errorf("no numeric values to correlate")
	}

	// Calculate means
	sum1, sum2 := 0.0, 0.0
	for i := 0; i < n; i++ {
		sum1 += vals1[i]
		sum2 += vals2[i]
	}
	mean1, mean2 := sum1/float64(n), sum2/float64(n)

	// Calculate correlation
	var num, den1, den2 float64
	for i := 0; i < n; i++ {
		d1 := vals1[i] - mean1
		d2 := vals2[i] - mean2
		num += d1 * d2
		den1 += d1 * d1
		den2 += d2 * d2
	}

	if den1 == 0 || den2 == 0 {
		return 0, nil
	}

	return num / math.Sqrt(den1*den2), nil
}

// Filter returns rows matching a condition
func (c *CSVReader) Filter(column string, op string, value any) ([]map[string]any, error) {
	if err := c.loadFile(); err != nil {
		return nil, err
	}

	colIndex := -1
	for i, h := range c.headers {
		if h == column {
			colIndex = i
			break
		}
	}
	if colIndex == -1 {
		return nil, fmt.Errorf("column '%s' not found", column)
	}

	var result []map[string]any
	for i := 1; i < len(c.lines); i++ {
		if strings.TrimSpace(c.lines[i]) == "" {
			continue
		}
		values := c.parseLine(c.lines[i])
		if colIndex >= len(values) {
			continue
		}

		cellValue := c.inferValue(values[colIndex])
		if matchesCondition(cellValue, op, value) {
			row := make(map[string]any)
			for j, header := range c.headers {
				if j < len(values) {
					row[header] = c.inferValue(values[j])
				} else {
					row[header] = ""
				}
			}
			result = append(result, row)
		}
	}
	return result, nil
}

// matchesCondition checks if a value matches a condition
func matchesCondition(cellValue any, op string, compareValue any) bool {
	switch op {
	case "==", "=", "eq":
		return fmt.Sprintf("%v", cellValue) == fmt.Sprintf("%v", compareValue)
	case "!=", "ne":
		return fmt.Sprintf("%v", cellValue) != fmt.Sprintf("%v", compareValue)
	case ">", "gt":
		return toFloatSimple(cellValue) > toFloatSimple(compareValue)
	case ">=", "gte":
		return toFloatSimple(cellValue) >= toFloatSimple(compareValue)
	case "<", "lt":
		return toFloatSimple(cellValue) < toFloatSimple(compareValue)
	case "<=", "lte":
		return toFloatSimple(cellValue) <= toFloatSimple(compareValue)
	case "contains":
		return strings.Contains(fmt.Sprintf("%v", cellValue), fmt.Sprintf("%v", compareValue))
	case "startswith":
		return strings.HasPrefix(fmt.Sprintf("%v", cellValue), fmt.Sprintf("%v", compareValue))
	case "endswith":
		return strings.HasSuffix(fmt.Sprintf("%v", cellValue), fmt.Sprintf("%v", compareValue))
	}
	return false
}

// toFloatSimple converts any value to float64 (for CSV analytics)
func toFloatSimple(v any) float64 {
	switch val := v.(type) {
	case float64:
		return val
	case int64:
		return float64(val)
	case int:
		return float64(val)
	case string:
		if f, err := strconv.ParseFloat(val, 64); err == nil {
			return f
		}
	}
	return 0
}

// Sort returns rows sorted by a column
func (c *CSVReader) Sort(column string, ascending bool) ([]map[string]any, error) {
	rows, err := c.ReadAll()
	if err != nil {
		return nil, err
	}

	// Simple bubble sort (could be optimized)
	for i := 0; i < len(rows)-1; i++ {
		for j := i + 1; j < len(rows); j++ {
			v1 := rows[i][column]
			v2 := rows[j][column]

			shouldSwap := false
			// Try numeric comparison first
			f1, f2 := toFloatSimple(v1), toFloatSimple(v2)
			if f1 != 0 || f2 != 0 {
				if ascending {
					shouldSwap = f1 > f2
				} else {
					shouldSwap = f1 < f2
				}
			} else {
				// String comparison
				s1, s2 := fmt.Sprintf("%v", v1), fmt.Sprintf("%v", v2)
				if ascending {
					shouldSwap = s1 > s2
				} else {
					shouldSwap = s1 < s2
				}
			}

			if shouldSwap {
				rows[i], rows[j] = rows[j], rows[i]
			}
		}
	}
	return rows, nil
}

// GroupBy groups rows by a column and applies aggregation
func (c *CSVReader) GroupBy(groupCol string, aggCol string, aggFunc string) ([]map[string]any, error) {
	rows, err := c.ReadAll()
	if err != nil {
		return nil, err
	}

	groups := make(map[string][]float64)
	for _, row := range rows {
		key := fmt.Sprintf("%v", row[groupCol])
		val := toFloatSimple(row[aggCol])
		groups[key] = append(groups[key], val)
	}

	var result []map[string]any
	for key, vals := range groups {
		var aggValue float64
		switch strings.ToLower(aggFunc) {
		case "sum":
			for _, v := range vals {
				aggValue += v
			}
		case "mean", "avg":
			for _, v := range vals {
				aggValue += v
			}
			aggValue /= float64(len(vals))
		case "min":
			aggValue = vals[0]
			for _, v := range vals {
				if v < aggValue {
					aggValue = v
				}
			}
		case "max":
			aggValue = vals[0]
			for _, v := range vals {
				if v > aggValue {
					aggValue = v
				}
			}
		case "count":
			aggValue = float64(len(vals))
		}

		result = append(result, map[string]any{
			groupCol: key,
			aggCol:   aggValue,
		})
	}
	return result, nil
}

// Summary returns a quick text summary of the data
func (c *CSVReader) Summary() (string, error) {
	if err := c.loadFile(); err != nil {
		return "", err
	}

	rowCount, _ := c.RowCount()
	colCount := len(c.headers)
	nulls, _ := c.NullCount()
	types, _ := c.DataTypes()

	var sb strings.Builder
	sb.WriteString(fmt.Sprintf("=== CSV Summary ===\n"))
	sb.WriteString(fmt.Sprintf("Rows: %d\n", rowCount))
	sb.WriteString(fmt.Sprintf("Columns: %d\n\n", colCount))
	sb.WriteString("Column Details:\n")

	for _, h := range c.headers {
		dtype := types[h]
		nullCount := nulls[h]
		stats, _ := c.Stats(h)

		sb.WriteString(fmt.Sprintf("\n  %s (%s):\n", h, dtype))
		sb.WriteString(fmt.Sprintf("    Nulls: %d\n", nullCount))

		if isNumeric, ok := stats["numeric"].(bool); ok && isNumeric {
			sb.WriteString(fmt.Sprintf("    Min: %.2f, Max: %.2f\n", stats["min"], stats["max"]))
			sb.WriteString(fmt.Sprintf("    Mean: %.2f, Median: %.2f\n", stats["mean"], stats["median"]))
		} else {
			unique := stats["unique"].(int)
			sb.WriteString(fmt.Sprintf("    Unique: %d\n", unique))
		}
	}

	return sb.String(), nil
}

// CSVWriter writes CSV files
type CSVWriter struct {
	file      *File
	delimiter string
	headers   []string
}

// NewCSVWriter creates a new CSV writer
func NewCSVWriter(path string, headers []string) (*CSVWriter, error) {
	file, err := OpenFile(path, "w")
	if err != nil {
		return nil, err
	}
	writer := &CSVWriter{
		file:      file,
		delimiter: ",",
		headers:   headers,
	}
	// Write headers
	file.WriteLine(strings.Join(headers, ","))
	return writer, nil
}

// SetDelimiter sets the CSV delimiter for writing
func (c *CSVWriter) SetDelimiter(delim string) {
	c.delimiter = delim
}

// WriteRow writes a row to the CSV
func (c *CSVWriter) WriteRow(values []any) error {
	strValues := make([]string, len(values))
	for i, v := range values {
		strValues[i] = c.escapeField(fmt.Sprintf("%v", v))
	}
	return c.file.WriteLine(strings.Join(strValues, c.delimiter))
}

// escapeField properly quotes and escapes CSV fields
func (c *CSVWriter) escapeField(field string) string {
	needsQuotes := strings.Contains(field, c.delimiter) ||
		strings.Contains(field, "\"") ||
		strings.Contains(field, "\n") ||
		strings.Contains(field, "\r")

	if !needsQuotes {
		return field
	}

	// Escape quotes by doubling them
	escaped := strings.ReplaceAll(field, "\"", "\"\"")
	return "\"" + escaped + "\""
}

// WriteRowMap writes a row from a map
func (c *CSVWriter) WriteRowMap(row map[string]any) error {
	values := make([]any, len(c.headers))
	for i, header := range c.headers {
		if v, ok := row[header]; ok {
			values[i] = v
		} else {
			values[i] = ""
		}
	}
	return c.WriteRow(values)
}

// Close closes the CSV writer
func (c *CSVWriter) Close() error {
	return c.file.Close()
}

// GetField extracts a field from any Petroleum object using reflection
// All Petroleum objects have a "fields" map[string]any field
func GetField(obj any, fieldName string) any {
	if obj == nil {
		return nil
	}

	// Use reflection to access the fields map
	v := reflect.ValueOf(obj)
	if v.Kind() == reflect.Ptr {
		v = v.Elem()
	}

	if v.Kind() != reflect.Struct {
		return nil
	}

	fieldsField := v.FieldByName("Fields")
	if !fieldsField.IsValid() {
		return nil
	}

	if fieldsMap, ok := fieldsField.Interface().(map[string]any); ok {
		return fieldsMap[fieldName]
	}

	return nil
}

// CallMethod calls a method on any Petroleum object using reflection
// Returns (result, error) like all Petroleum methods
func CallMethod(obj any, methodName string, ctx *Ctx, args ...any) (any, error) {
	if obj == nil {
		return nil, fmt.Errorf("cannot call method on nil")
	}

	v := reflect.ValueOf(obj)
	method := v.MethodByName(methodName)
	if !method.IsValid() {
		return nil, fmt.Errorf("method %s not found", methodName)
	}

	// Build argument list: ctx first, then other args
	callArgs := make([]reflect.Value, 1+len(args))
	callArgs[0] = reflect.ValueOf(ctx)
	for i, arg := range args {
		callArgs[i+1] = reflect.ValueOf(arg)
	}

	results := method.Call(callArgs)
	if len(results) == 2 {
		var err error
		if results[1].Interface() != nil {
			err = results[1].Interface().(error)
		}
		return results[0].Interface(), err
	}

	return nil, nil
}

// ToInt converts any value to an integer
func ToInt(val any) (any, error) {
	switch v := val.(type) {
	case int64:
		return v, nil
	case int:
		return int64(v), nil
	case float64:
		return int64(v), nil
	case string:
		var result int64
		_, err := fmt.Sscanf(v, "%d", &result)
		if err != nil {
			return nil, fmt.Errorf("cannot convert '%s' to int", v)
		}
		return result, nil
	case bool:
		if v {
			return int64(1), nil
		}
		return int64(0), nil
	default:
		return nil, fmt.Errorf("cannot convert %T to int", val)
	}
}

// ToFloat converts any value to a float
func ToFloat(val any) (any, error) {
	switch v := val.(type) {
	case float64:
		return v, nil
	case int64:
		return float64(v), nil
	case int:
		return float64(v), nil
	case string:
		var result float64
		_, err := fmt.Sscanf(v, "%f", &result)
		if err != nil {
			return nil, fmt.Errorf("cannot convert '%s' to float", v)
		}
		return result, nil
	default:
		return nil, fmt.Errorf("cannot convert %T to float", val)
	}
}

// ToStr converts any value to a string
func ToStr(val any) (any, error) {
	if val == nil {
		return "None", nil
	}
	return fmt.Sprintf("%v", val), nil
}

// ToBool converts any value to a boolean
func ToBool(val any) (any, error) {
	switch v := val.(type) {
	case bool:
		return v, nil
	case int64:
		return v != 0, nil
	case int:
		return v != 0, nil
	case float64:
		return v != 0, nil
	case string:
		return v != "", nil
	case []any:
		return len(v) > 0, nil
	case map[string]any:
		return len(v) > 0, nil
	case nil:
		return false, nil
	default:
		return true, nil
	}
}
